ScriptName RealFuel Extends ScriptObject

; You can use these however you like as long as credit is given.
; If you want to call them directly, simply run RealFuel.TheFunction() or Import RealFuel at the beginning of your script.

; This will prompt Real Fuel to perform all checks for travel and damage systems.
; It should generally be called after any alterations to SpaceshipGravJumpFuel's value.
bool Function Dashboard() Global
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        RF_FuelHandler.Dashboard()
        RF_FuelHandler.QuickTrace("Daskboard called externally")
        Return True
    Else
        Return False
    EndIf
EndFunction

; This will refuel the player's ship.
; Return Values:
; -1 = exception
;  0 = all fuel was used
; +1 = more fuel was added than the ship could hold
float Function RefuelPlayerShip(float FuelToAdd, bool IsTotalRefill) Global
    float FuelDiff = -1
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        RF_FuelHandler.GetShip()
        float FuelDrawn = RF_FuelHandler.RealFuel(abComplete = IsTotalRefill, afAmount = FuelToAdd)
        FuelDiff = FuelToAdd - FuelDrawn
    EndIf
    Return FuelDiff
EndFunction

; This will drain fuel from the player's ship.
; Currently it will only drain either a set amount or the entire tank.
; There is a smart cap to avoid draining the value below 0, so don't worry about the drain amount.
; Return Values:
; -1 = exception
;  0 = script error
;  other = fuel in ship after drain procedure
float Function DrainPlayerShip(float FuelToDrain, bool IsTotalDrain) Global
    float FuelRemaining = -1
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        RF_FuelHandler.GetShip()
        RF_FuelHandler.ForceDrainFuel(abComplete = IsTotalDrain, afAmount = FuelToDrain)
        RF_FuelHandler.Dashboard()
        FuelRemaining = RF_FuelHandler.CheckFuelAmount()
    EndIf
    Return FuelRemaining
EndFunction

; This returns a float value (0.0 > 1.0) of the fuel in the player ship's tanks.
; If AsPercent = true, it will return 0 - 100.
; -1 return value indicates the function failed to run.
float Function GetTankLevels(bool abAsPercent) Global
    float fuel = 1.0
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        RF_FuelHandler.GetShip()
        If !abAsPercent
            fuel = RF_FuelHandler.GetFuelLevel()
        Else
            fuel = RF_FuelHandler.GetFuelLevelInt() as float
        EndIF
    Else
        Fuel = -1
    EndIF
    Return Fuel
EndFunction

; This will prompt the main system to re-assign its used ship to SQ_PlayerShip.PlayerShip
; You can either call this for convenience or if you changed the ship forcibly somehow
; returns true if assignment was successful
bool Function ResetShipRF() Global
    bool IsShipOK = false
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    SQ_PlayerShipScript SQ_PlayerShip = Game.GetForm(0x000174a2) as SQ_PlayerShipScript
    If RF_FuelHandler
        SpaceshipReference MyShipRF = RF_FuelHandler.GetShip()
        SpaceshipReference MyShipSQ = SQ_PlayerShip.PlayerShip.GetShipRef()
        If MyShipRF == MyShipSQ
            IsShipOK = True
        EndIF
    EndIF
    Return IsShipOK
EndFunction

; Imports a FormList to an array.
; Mostly useful for performance: Formlists are convenient to edit but very slow to call.
; If Accurate = true, Form[42] will be FormList Entry #42. 0 & 1 will *both* return the first Form.
; Note that you can cast the output to any type with e.g. PotionList = ImportFormList(MyFormList) as Potion[]
Form[] Function ImportFormlist(FormList asTheList, bool abAccurate = false) Global
    Int listSize = asTheList.GetSize()
    Form[] Output = new Form[listSize]
    If abAccurate
        listSize += 1
        Output[0] = asTheList.GetAt(0)
        Int i = 0
        While i < listSize
            Output[i + 1] = asTheList.GetAt(i) as Form
            i += 1
        EndWhile
    Else
        Int i = 0
        While i < listSize
            Output[i] = asTheList.GetAt(i) as Form
            i += 1
        EndWhile
    EndIF
    Return Output
EndFunction

; rbt
; Gets the *location* of a Location's associated Planet.
; This will return e.g. Cydonia will return SSolPMars.
; May or may not work in space depending on the cell.
; # Does NOT return the Planet record!
Location Function GetPlanetaryBody(Location akLocation) Global
    Keyword LocTypeMajorOrbital = Game.GetFormFromFile(0x00070A54, "Starfield.esm") as Keyword
    Location[] Planets = akLocation.GetParentLocations(LocTypeMajorOrbital)
    Location WeAreOn = Planets[0]
    Return WeAreOn
EndFunction

; rbt
; Returns the Location reference of the specified Location's Star System. 
; This will return e.g. Mars or Deimos will both return SSol.
Location Function GetStarSystem(Location akLocation) Global
    Keyword LocTypeStarSystem = Game.GetFormFromFile(0x0000149F, "Starfield.esm") as Keyword
    Location[] System = akLocation.GetParentLocations(LocTypeStarSystem)
    Location SystemWeAreIn = System[0]
    Return SystemWeAreIn
EndFunction

; rbt
; Gets the *location* of the planet the Player is on.
; This will return e.g. SSolPMars if the player is in Cydonia.
; May or may not work in space depending on the cell.
; # Does NOT return the Planet record!
Location Function GetPlayerCurrentPlanet() Global
    Location PlayerIsIn = Game.GetPlayer().GetCurrentLocation()
    Keyword LocTypeMajorOrbital = Game.GetFormFromFile(0x00070A54, "Starfield.esm") as Keyword
    Location[] Planets = PlayerIsIn.GetParentLocations(LocTypeMajorOrbital)
    Location PlayerIsOn = Planets[0]
    Return PlayerIsOn
EndFunction

; rbt
; Returns the Location of the Star System the player is in currently.
; This will return e.g. SSol if the Player is on Mars or Deimos.
Location Function GetPlayerCurrentSystem() Global
    Location PlayerIsIn = Game.GetPlayer().GetCurrentLocation()
    Keyword LocTypeStarSystem = Game.GetFormFromFile(0x0000149F, "Starfield.esm") as Keyword
    Location[] System = PlayerIsIn.GetParentLocations(LocTypeStarSystem)
    Location SystemWeAreIn = System[0]
    Return SystemWeAreIn
EndFunction

; rbt
; Determines if the specified Planet is Settled or not.
; May or may not work in space depending on the cell.
bool Function IsPlanetSettled(Location akLocation) Global
    bool Settled = false
    Keyword LocTypeMajorOrbital = Game.GetFormFromFile(0x00070A54, "Starfield.esm") as Keyword
    Keyword LocTypeSettledPlanet = Game.GetFormFromFile(0x00062F2F, "Starfield.esm") as Keyword
    Location[] Orbital = akLocation.GetParentLocations(LocTypeMajorOrbital)
    If Orbital[0].HasKeyword(LocTypeSettledPlanet)
        Settled = True
    EndIF
    Return Settled
EndFunction

; rbt
; Determines if the specified System is Settled or not.
bool Function IsSystemSettled(Location akLocation) Global
    bool Settled = false
    Keyword LocTypeStarSystem = Game.GetFormFromFile(0x0000149F, "Starfield.esm") as Keyword
    Keyword LocTypeSettledSystem = Game.GetFormFromFile(0x00061990, "Starfield.esm") as Keyword
    Location[] System = akLocation.GetParentLocations(LocTypeStarSystem)
    If System[0].HasKeyword(LocTypeSettledSystem)
        Settled = True
    EndIF
    Return Settled
EndFunction

; rbt
; Round to any multiple of an integer.
; This will return 0 if your input is lower than the Lasso value.
int Function RoundToMultiple(float afToRound, int aiLasso = 10) Global
    int Round1 = 0
    If afToRound > aiLasso
        Round1 = afToRound as int
        Round1 += ( aiLasso - ( Round1 % aiLasso ) )
    Else
        Round1 = 0
    EndIF
    Return Round1
EndFunction

; rbt
; Checks if two indices are neighboring *and* comprise an even pair in an array starting from 0
; e.g. 4 and 5 will return true, but 5 and 6 will return false.
bool Function GetIsValidPair(int x, int y) Global
    bool pair = false
    int diff = Math.Abs(x - y) as int
    int minVal = Math.Min(x, y) as int
    if diff == 1 && (minVal % 2) == 0
        pair = true
    endif
    return pair
EndFunction

; rbt
; Allows you to add weight to a random (initially fair) roll.
; 'Wins' become more likely as Weight is increased and vice versa.
bool Function WeightedDieRoll(int aiWeight = 0) Global
    bool WonTheRoll = false
    If ( 50 + aiWeight ) > Utility.RandomInt(0, 100)
        WonTheRoll = true
    EndIf
    Return WonTheRoll
EndFunction

; rbt
; Returns 1 +- a random amount capped at Max.
float Function GetVariance(float afMax = 0.25) Global
    bool Random = Game.GetDieRollSuccess(50)
    float Divergence = 1.0
    float RandomAmount = Utility.RandomFloat(0.0, afMax)
    If Random
        Divergence = 1.0 - Math.Clamp( RandomAmount, 0, 0.99 )
    Else
        Divergence = 1.0 + RandomAmount
    EndIF
    return Divergence
EndFunction

; rbt
; Returns the Player Ship from SQ_PlayerShip.
; If Home = true, returns the Home Ship.
SpaceshipReference Function GetPlayerShip(bool abHome = false) Global
    SQ_PlayerShipScript SQ_PlayerShip = Game.GetForm(0x000174a2) as SQ_PlayerShipScript
    SpaceshipReference TheShip
    If !abHome
        TheShip = SQ_PlayerShip.PlayerShip.GetShipRef()
    Else
        TheShip = SQ_PlayerShip.HomeShip.GetShipRef()
    EndIf
    Return TheShip
EndFunction

; rbt
; Shorthand to get player ship without typing a bunch
SpaceshipReference Function GPS() Global
    Return RealFuel.GetPlayerShip(false)
EndFunction

; rbt
; Checks if the player is on their own ship
; Useful because vanilla CNDFs only check if the player is on *a* ship.
bool Function IsPlayerOnShip() Global
    bool OnShip = False
    SQ_PlayerShipScript SQ_PlayerShip = Game.GetForm(0x000174a2) as SQ_PlayerShipScript
    If Game.GetPlayer().GetCurrentLocation() == Sq_PlayerShip.PlayerShipInteriorLocation.GetLocation()
        OnShip = True
    EndIf
    Return OnShip
EndFunction

; rbt
; Remove a set percentage of the passed in Form from a Spaceship's hold.
; Returns the count of items removed.
int Function ForceJettisonPercent(Form asFormtoRemove, float afAmount = 0.5, SpaceshipReference akSpaceship = None) Global
    int AmountInHold = akSpaceship.GetItemCount(asFormtoRemove)
    float PercentToDrop = Math.Clamp(afAmount, 0, 1.0)
    int JettisonCount = Math.Ceiling( AmountInHold * PercentToDrop )
    akSpaceship.RemoveItem(asFormtoRemove, JettisonCount, true)
    Return JettisonCount
EndFunction

; rbt
; Tries to collect the specified amount of Credits from the player.
; If Seizure is true, it will take whatever the player *did* have.
; Return values:
;   0 = Player had enough credits, they were taken.
;   1+ = The player was short by the return value.
int Function TryToPayCredits(int aiCredits, bool abSeizure = False, bool abSilent = true) Global
	int Remainder = 0
    Actor PlayerRef = Game.GetPlayer()
    MiscObject Credits = Game.GetCredits()
	Int CreditsPlayerHas = PlayerRef.GetItemCount(Credits)
	If CreditsPlayerHas >= aiCredits
		PlayerRef.RemoveItem(Credits, aiCredits, abSilent)
	Else
		Remainder = aiCredits - CreditsPlayerHas
		If abSeizure
			PlayerRef.RemoveItem(Credits, CreditsPlayerHas, abSilent)
		EndIf
	EndIf
	return Remainder
EndFunction

; rbt
; Reduce Actor's AV by a set percentage of its current value - default is 50%.
; This is useful to avoid e.g. killing the actor while massively draining health.
; If Safe = false, it will not cap and can bring the value negative.
; Returns the amount the AV was reduced by.
float Function DamageValueSafe(SpaceshipReference akActor, ActorValue ValueToDamage, float afAmountToDamage = 0.5, bool abSafe = True) Global
    float DamageToDo = 0
    float CurrAV = akActor.GetValue(ValueToDamage)
    If !abSafe
        DamageToDo = CurrAV * afAmountToDamage
    Else
        DamageToDo = Math.Clamp( ( CurrAV * afAmountToDamage ), 0, ( CurrAV * 0.99 ) )
    EndIf
    akActor.DamageValue(ValueToDamage, DamageToDo)
    Return DamageToDo
EndFunction

; Return the value of MyShip or TheShip
float Function GV(ActorValue akValue, SpaceshipReference akTheShip = None) Global
    float Value = 1.0
    IF akTheShip
        Value = akTheShip.GetValue(akValue)
    Else
        Value = RealFuel.GPS().GetValue(akValue)
    EndIf
    Return Value
EndFunction

; Return the value % as int of MyShip or TheShip with optional rounding to tens
int Function GVPCT(ActorValue akValue, bool abModTens = false, SpaceshipReference akTheShip = None) Global
    SpaceshipReference ShipToCheck
    int Percent = -1
    If akTheShip
        ShipToCheck = akTheShip
    Else
        ShipToCheck = RealFuel.GPS()
    EndIF
    float Base = GBV(akValue, ShipToCheck)
    float Now = GV(akValue, ShipToCheck)
    Percent = Math.Round( Now / Base )
    If abModTens 
        If Percent > 10 && Percent < 90
            Percent += ( 10 - ( Percent % 10 ) )
        EndIf
    EndIF
    Return Percent
EndFunction

; Return the base value of Myship or TheShip
float function GBV(ActorValue akValue, SpaceshipReference akTheShip = None) Global
    float Value = 2.0
    IF akTheShip
        Value = akTheShip.GetBaseValue(akValue)
    Else
        Value = RealFuel.GPS().GetBaseValue(akValue)
    EndIf
    Return Value
EndFunction

; rbt
; Return the input value rounded to Accuracy
; Rounds to 1% and 100% by default unless AllowZero = true
int Function GetDisplayPercent(int aiToRound, int aiAccuracy = 5, bool AllowZero = false) Global
    int Output = -1
    int Input = Math.Clamp(aiToRound, 0, 100) as int
    If Math.abs(100 - Input) <= aiAccuracy ; We are less than A away from 100
        Return 100
    ElseIf Math.abs(Input) <= aiAccuracy ; We are less than A away from 0
        If AllowZero
            Return 0
        Else
            Return 1
        EndIF
    Else
        Output = aiToRound
    EndIF
    Output += ( aiAccuracy - ( Output % aiAccuracy ) )
    Return Output
EndFunction

; these are here purely to avoid the hellscape of script fragment properties in CK
bool Function CallCargoPanelFunction(ObjectReference akTargetRef) Global
    bool Ran = false
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        Ran = True
        RF_FuelHandler.HandleActivateCargoPanel(akTargetRef)
    EndIf
    Return Ran
EndFunction

; these are here purely to avoid the hellscape of script fragment properties in CK
bool Function CallContainerRefuelFunction(ObjectReference akTargetRef) Global
    bool Ran = false
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        Ran = True
        RF_FuelHandler.HandleActivateFuelContainer(akTargetRef)
    EndIf
    Return Ran
EndFunction

; these are here purely to avoid the hellscape of script fragment properties in CK
bool Function CallRFPanelRefuelFunction(ObjectReference akTargetRef) Global
    bool Ran = false
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        Ran = True
        RF_FuelHandler.HandleRefuelPanel()
    EndIf
    Return Ran
EndFunction

; these are here purely to avoid the hellscape of script fragment properties in CK
bool Function CallRFPanelSiphonFunction(ObjectReference akTargetRef) Global
    bool Ran = false
    RFNF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel_RE.esm") as RFNF:FuelHandlerQuest
    If RF_FuelHandler
        Ran = True
        RF_FuelHandler.HandleSiphonPanel()
    EndIf
    Return Ran
EndFunction

; This is called to disable travel restrictions on nav console activate
; deprecated with CNDF method
; bool Function CallStarMap() Global
;     RFNF:TravelHandlerQuest RF_TravelHandler = Game.GetFormFromFile(0x000829, "rbt_RealFuel_RE.esm") as RFNF:TravelHandlerQuest
;     If RF_TravelHandler
;         RF_TravelHandler.HandleMapActivate()
;         Return True
;     Else
;         Return False
;     EndIf
; EndFunction